
/************************************************************************
 *
 * \file: AoapTransport.h
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto - Prototype
 *
 * \author: D. Girnus / ADIT/SW1/Brunel / dgirnus@de.adit-jv.com
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#ifndef AAUTO_AOAP_TRANSPORT_H
#define AAUTO_AOAP_TRANSPORT_H


/* *************  includes  ************* */

#include <aauto/util/Transport.h>
#include <semaphore.h>

/* *************  defines  ************* */

namespace adit { namespace aauto {

/**
 * @struct aoapAccessoryInfo_t
 * The accessory information consisting of the six identification strings
 * and a flag for enabling audio
 */
struct aoapAccessoryInfo_t
{
    const char *manufacturer; // The manufacturer of the MD application (for identification of Android application)
    const char *modelName;    // The model name of the MD application (for identification of Android application)
    const char *description;  // The description of the MD application (for identification of Android application)
    const char *version;      // The version (for identification of Android application)
    const char *uri;          // The URI where to download the MD application (for identification of Android application)
    const char *serial;       // The serial number of the HU
    unsigned int enableAudio; // =0 : disable audio. >0 : enable audio (only necessary for AOA 2.0)
};

/**
 * @struct aoapDeviceInfo_t
 * The device information (vendorId, producId, serial number) to identify the device
 */
struct aoapDeviceInfo_t
{
    uint32_t vendorId;
    uint32_t productId;
    char* pSerial;

    aoapAccessoryInfo_t aoapAccessoryInfo;
};

/**
 * @struct aoapTransportInfo_t
 * The AOAP information which are necessary for AoapTransport e.g. GalReceiver Transport
 */
struct aoapTransportInfo_t
{
    int32_t aoapAccessoryId;
    int32_t aoapDeviceId;
};


/* class AoapDevice
 * This class provides a member function to switch the MD into the accessory mode using the AOAP
 * library.
 */
class AoapDevice {
public:
    AoapDevice(){
        mSwitchResult = -1;
        if (sem_init(&mAoapSyncSem, 0, 0) == -1) {
            assert(&mAoapSyncSem);
        }
    };
    virtual ~AoapDevice() {
        sem_post(&mAoapSyncSem);
        sem_destroy(&mAoapSyncSem);
    };
    /**
     * @brief Initialize the AOAP library and switch the device identified by aoapDeviceInfo_t
     * to the accessory mode. It return in structure aoapTransportInfo_t the necessary information
     * to use the AOAP library transport by class AoapTransport.
     * This API is blocking and return if the device was switched of the timeout set by
     * inTimeout passed.
     *
     * @param inAoapDeviceInfo structure with device information and accessory information
     * @param outTransportInfo parameters which a necessary to use the AOAP library as transport
     *        (class AoapTransport)
     * @param inTimeout[ms] maximum time in millisecond to wait to switch the device to accessory mode
     *
     * @return 0 if success, otherwise negative return value
     */
    int switchDevice(aoapDeviceInfo_t* inAoapDeviceInfo, aoapTransportInfo_t* outTransportInfo,
            unsigned int inTimeout);
private:
    /**
     * @brief Callback for AOAP library which is triggered during/after switch to accessory mode
     */
    static void switchCallback(int accessoryId, int deviceId, int result, void *pToken,
            unsigned int audioSupport);

    sem_t mAoapSyncSem;
    int mSwitchResult;
};


/* This class represents the AOAP transport.
 * The protected methods overload the GalReceiver::Transport methods by the AOAP library.
 */
class AoapTransport : public Transport {
public:
    /**
     * @brief Initialize the AoapTransport and uses later the parameters of
     * inAoapTransportInfo for the AOAP library transport e.g. for GalReceiver::Transport.
     *
     * @param inReceiver pointer to the GalReceiver.
     * @param inAoapTransportInfo parameters which a necessary to use the AOAP library transport
     *        (class AoapTransport)
     */
    AoapTransport(shared_ptr<GalReceiver> inReceiver, aoapTransportInfo_t* inAoapTransportInfo);
    ~AoapTransport();

protected:
    void abortReads();
    int read(void* buf, size_t len);
    int write(void* buf, size_t len);

private:
    /**
     * @brief Validates the return value of read/write transfer represented by 'inResult'
     * and notifies transport status via GalReceiver::Transport::notifyStatus.
     *
     * @param inResult Return value of read/write request.
     * @param inWasWriteRequest Set to true if transfer was a write request. Otherwise to false.
     *
     * @return 0 if success. TRANSPORT_ERROR_NONFATAL in case of I/O error or timeout.
     *         TRANSPORT_ERROR_FATAL in case of fatal errors.
     */
    int32_t validateTransfer(int32_t inResult, bool inWasWriteRequest);

    /* AOAP library parameter */
    /* to identify the MD inside the AOAP library */
    int32_t mAoapAccId;
    int32_t mAoapDevId;

    /**
     * @struct aoapReadBuffer_t
     * Members to handle local read buffer.
     * */
    struct aoapReadBuffer_t {
        unsigned char* buf;
        size_t size;
        size_t pos;
        size_t bytesRemaining;
    };
    aoapReadBuffer_t mReadBuffer;
};


} } /* namespace adit { namespace aauto { */


#endif /* AAUTO_AOAP_TRANSPORT_H */
